#region References

using System;
using System.Collections;
using System.Data;
using System.Text;

using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

using PATIENT = gov.va.med.vbecs.Common.VbecsTables.Patient;
using PATIENT_CHANGE = gov.va.med.vbecs.Common.VbecsTables.PatientChange;
using PATIENT_LOCATION = gov.va.med.vbecs.Common.VbecsTables.PatientLocation;
using SProc = gov.va.med.vbecs.Common.VbecsStoredProcs;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.Parsers
{
    #region Header

    //<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
    //<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
    //<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
    //<Developers>
    //	<Developer>Brian Tomlin</Developer>
    // <Developer>David Askew</Developer>
    //</Developers>
    //<SiteName>Hines OIFO</SiteName>
    //<CreationDate>3/27/2004</CreationDate>
    //<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
    //<summary>This class is responsible for handling incoming HL7 messages from VistA when patient demographics that VBECS is interested in are changed in VistA.</summary>

    #endregion

    /// <summary>
    /// Class re-written to support new patient update interface (CR 2960)
    /// CR 2985: Updated to provide better error handling of missing\invalid message data
    /// </summary>
    public sealed class PatientUpdateHL7Parser
    {
        #region Inner Classes

        /// <summary>
        /// Used to hold data parsed from ADT message
        /// </summary>
        private sealed class PatientUpdateData
        {
            #region Variables

            private string _locationName;
            private string _locationRoomBed;
            private string _locationDivisionCode;

            private string _operatorId;

            private string _patientFirstName;
            private string _patientDobCode;
            private string _patientIcn;
            private string _patientLastName;
            private string _patientMiddleName;
            private string _patientSexCode;
            private string _patientSsn;
            private string _patientDfn;

            private DateTime _patientDob;
            private DateTime _patientDod;

            private bool _inpatientIndicator;

            private Guid _patientGuid;

            #endregion

            #region Properties

            /// <summary>
            /// 
            /// </summary>
            internal bool InpatientIndicator
            {
                get
                {
                    return this._inpatientIndicator;
                }
                set
                {
                    this._inpatientIndicator = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string LocationName
            {
                get
                {
                    return this._locationName;
                }
                set
                {
                    this._locationName = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string LocationRoomBed
            {
                get
                {
                    return this._locationRoomBed;
                }
                set
                {
                    this._locationRoomBed = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string LocationDivisionCode
            {
                get
                {
                    return this._locationDivisionCode;
                }
                set
                {
                    this._locationDivisionCode = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string OperatorId
            {
                get
                {
                    return this._operatorId;
                }
                set
                {
                    this._operatorId = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string PatientDfn
            {
                get
                {
                    return this._patientDfn;
                }
                set
                {
                    this._patientDfn = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal DateTime PatientDob
            {
                get
                {
                    return this._patientDob;
                }
                set
                {
                    this._patientDob = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string PatientDobCode
            {
                get
                {
                    return this._patientDobCode;
                }
                set
                {
                    this._patientDobCode = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal DateTime PatientDod
            {
                get
                {
                    return this._patientDod;
                }
                set
                {
                    this._patientDod = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal Guid PatientGuid
            {
                get
                {
                    return this._patientGuid;
                }
                set
                {
                    this._patientGuid = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string PatientFirstName
            {
                get
                {
                    return this._patientFirstName;
                }
                set
                {
                    this._patientFirstName = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string PatientIcn
            {
                get
                {
                    return this._patientIcn;
                }
                set
                {
                    this._patientIcn = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string PatientLastName
            {
                get
                {
                    return this._patientLastName;
                }
                set
                {
                    this._patientLastName = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string PatientMiddleName
            {
                get
                {
                    return this._patientMiddleName;
                }
                set
                {
                    this._patientMiddleName = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string PatientSexCode
            {
                get
                {
                    return this._patientSexCode;
                }
                set
                {
                    this._patientSexCode = value;
                }
            }

            /// <summary>
            /// 
            /// </summary>
            internal string PatientSsn
            {
                get
                {
                    return this._patientSsn;
                }
                set
                {
                    this._patientSsn = value;
                }
            }

            #endregion
        }

        /// <summary>
        /// Used to generate response to ADT HL7 messages 
        /// </summary>
        private sealed class AdtAckMessage
        {
            #region Variables

            private const int MAX_ERROR_MESSAGE_SIZE = 300;

            private const string ACK_MESSAGE_TYPE = "ACK";
            private const string CARRIAGE_RETURN = "\x0D";

            #endregion

            #region HL7 Acknowledgement Methods

            /// <summary>
            ///  
            /// </summary>
            internal static HL7AckMessage BuildApplicationAcknowledgement(HL7Interface hl7Interface, AckCodes ackCode, string messageControlId, string errSegments, string triggerEvent)
            {
                StringBuilder ack = new StringBuilder();
                //
                ack.Append(BuildMSHSegment(hl7Interface, triggerEvent));
                ack.Append(BuildMSASegment(hl7Interface.FieldSeparator, ackCode, messageControlId));
                //
                if (errSegments != null && errSegments.Length > 0)
                {
                    ack.Append(errSegments);
                }
                //
                return new HL7AckMessage(ack.ToString());
            }

            /// <summary>
            /// 
            /// </summary>
            private static StringBuilder BuildMSHSegment(HL7Interface hl7Interface, string triggerEvent)
            {
                StringBuilder msh = new StringBuilder();
                //
                msh.Append("MSH");																	// MSH Segment ID
                msh.Append(hl7Interface.FieldSeparator);											// Field Separator
                msh.Append(hl7Interface.EncodingCharacters);										// Encoding Characters
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(hl7Interface.VbecsApplicationId.Trim());									// Sending Application
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(hl7Interface.VbecsFacilityId.Trim());									// Sending Facility
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(hl7Interface.InterfaceApplicationId.Trim());								// Receiving Application
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(hl7Interface.InterfaceFacilityId.Trim());								// Receiving Facility
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(System.DateTime.Now.ToString("yyyyMMddhhmmsszzz").Replace(":", ""));		// HL7 DateTime
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(ACK_MESSAGE_TYPE);														// ACK Message Type
                msh.Append(hl7Interface.EncodingCharacters[0]);
                msh.Append(triggerEvent);															// Trigger Event
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append("VBECS").Append(System.DateTime.Now.ToString("yMMddhhssffff"));
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(hl7Interface.ProcessingId);												// Processing ID
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(hl7Interface.VersionId.Trim());											// HL7 Version ID
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append("NE");
                msh.Append(hl7Interface.FieldSeparator);
                msh.Append("AL");
                msh.Append(CARRIAGE_RETURN);
                //
                return msh;
            }

            /// <summary>
            /// 
            /// </summary>
            private static StringBuilder BuildMSASegment(string fieldSeparator, AckCodes ackCode, string messageControlId)
            {
                StringBuilder msa = new StringBuilder();
                //
                msa.Append("MSA");
                msa.Append(fieldSeparator);
                msa.Append(ackCode);
                msa.Append(fieldSeparator);
                msa.Append(messageControlId);
                msa.Append(CARRIAGE_RETURN);
                //
                return msa;
            }

            /// <summary>
            /// Builds HL7 version 2.3 compliant ERR Segment		
            /// </summary>
            internal static string BuildErrSegment(
                string fieldSeparator,
                char componentSeparator,
                string segmentID,
                int segmentSequence,
                int fieldPosition,
                AckErrorCodes ackErrorCode,
                string errorMessage)
            {
                StringBuilder errSegment = new StringBuilder();
                //
                errSegment.Append("ERR");
                errSegment.Append(fieldSeparator);		// ERR.1
                // ******************************************************************************************************************
                // ERR.1.1: segment ID (Indicates the Segment location of the element that generated a parsing error.  
                // Specifies the 3-letter name for the segment where the error occured.)
                if (segmentID != null)
                {
                    errSegment.Append(segmentID);
                }
                errSegment.Append(componentSeparator);
                // ******************************************************************************************************************
                // ERR.	1.2: sequence (Indicates the Sequence of the element that generated a parsing error: Identifies the 
                // segment occurrence within the message.)
                errSegment.Append(segmentSequence);
                errSegment.Append(componentSeparator);
                // ******************************************************************************************************************
                // ERR.1.3: field position (Indicates the Field Position of the element that generated a parsing error.)
                if (fieldPosition > 0)
                {
                    errSegment.Append(fieldPosition);
                }
                errSegment.Append(componentSeparator);
                // ******************************************************************************************************************
                // ERR.1.4.1: code identifying error, identifier; required (Error Code from table 0357.)
                errSegment.Append(HL7Utility.GetAckErrorCodeFromEnum(ackErrorCode));
                //
                errSegment.Append(componentSeparator);
                // ******************************************************************************************************************
                // ERR.1.4.2: code identifying error, text; optional (Error Code Description from table 0357.)
                if (errorMessage != null && errorMessage != string.Empty)
                {
                    errSegment.Append(errorMessage.Length > MAX_ERROR_MESSAGE_SIZE ? errorMessage.Substring(0, MAX_ERROR_MESSAGE_SIZE) : errorMessage);
                }
                //
                errSegment.Append(CARRIAGE_RETURN);
                //
                return errSegment.ToString();
            }

            #endregion
        }

        #endregion

        #region Variables

        private const string EMPTY_LOCATION_1 = "''";
        private const string EMPTY_LOCATION_2 = "\"\"";
        private const string LAST_UPDATE_USER_VALUE = "VAFC PIMS";

        /// <summary>
        /// Generic error message template
        /// </summary>
        private const string ERROR_MESSAGE_BASE = "Error processing HL7 message: \n\n";
        /// <summary>
        /// HL7 error message template
        /// </summary>
        private const string ERROR_MESSAGE_HL7 = "Missing or invalid content in HL7 message: \n\n";
        /// <summary>
        /// Generic no-details error message
        /// </summary>
        private const string ERROR_MESSAGE_NO_DETAILS = "No details available.";

        // Events Logger
        private static readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

        #endregion

        #region Public Methods

        ///<Developers>
        ///	<Developer>David Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>5/23/2008</CreationDate>
        ///<TestCases>
        ///
        ///<Case type="0" testid ="5720"> 
        ///		<ExpectedInput>Valid HL7Interface, ADT A01 (Admit a Patient) HL7 message for a VBECS patient</ExpectedInput>
        ///		<ExpectedOutput>HL7ProtocolMessage with AA ack code</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="0" testid ="5721"> 
        ///		<ExpectedInput>Valid HL7Interface, ADT A02 (Transfer a Patient) HL7 message for a VBECS patient</ExpectedInput>
        ///		<ExpectedOutput>HL7ProtocolMessage with AA ack code</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="0" testid ="5750"> 
        ///		<ExpectedInput>Valid HL7Interface, ADT A08 (Patient Demographic Update\Appointment Check-In\Edit Patient Movement) HL7 message for a VBECS patient</ExpectedInput>
        ///		<ExpectedOutput>HL7ProtocolMessage with AA ack code</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="0" testid ="5751"> 
        ///		<ExpectedInput>Valid HL7Interface, ADT A03 (Discharge a Patient) HL7 message for a VBECS patient</ExpectedInput>
        ///		<ExpectedOutput>Valid HL7ProtocolMessage with AA ack code</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="0" testid ="5752"> 
        ///		<ExpectedInput>Valid HL7Interface, ADT A13 (Delete Discharge) HL7 message for a VBECS patient</ExpectedInput>
        ///		<ExpectedOutput>Valid HL7ProtocolMessage with AA ack code</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="0" testid ="5753"> 
        ///		<ExpectedInput>Valid HL7Interface, ADT A12 (Delete Transfer) HL7 message for a VBECS patient</ExpectedInput>
        ///		<ExpectedOutput>Valid HL7ProtocolMessage with AA ack code</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="0" testid ="5754"> 
        ///		<ExpectedInput>Valid HL7Interface, ADT A11 (Delete Admission) HL7 message for a VBECS patient</ExpectedInput>
        ///		<ExpectedOutput>Valid HL7ProtocolMessage with AA ack code</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="5931"> 
        ///		<ExpectedInput>Null HL7Interface, Valid HL7 message string</ExpectedInput>
        ///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="5932"> 
        ///		<ExpectedInput>Valid HL7Interface, Null HL7 message string</ExpectedInput>
        ///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="5935"> 
        ///		<ExpectedInput>Null HL7Interface, Null HL7 message string</ExpectedInput>
        ///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="5936"> 
        ///		<ExpectedInput>Invalid field separator (MSH[1])</ExpectedInput>
        ///		<ExpectedOutput>HL7Exception</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="6971"> 
        ///		<ExpectedInput>Invalid encoding characters (MSH[2]) - char[0]</ExpectedInput>
        ///		<ExpectedOutput>HL7Exception</ExpectedOutput>
        ///	</Case>
        ///	
        ///<Case type="1" testid ="8914"> 
        ///		<ExpectedInput>Invalid encoding characters (MSH[2]) - char[1]</ExpectedInput>
        ///		<ExpectedOutput>HL7Exception</ExpectedOutput>
        ///	</Case>
        ///	
        ///<Case type="1" testid ="8916"> 
        ///		<ExpectedInput>Invalid encoding characters (MSH[2]) - char[2]</ExpectedInput>
        ///		<ExpectedOutput>HL7Exception</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="8917"> 
        ///		<ExpectedInput>Invalid encoding characters (MSH[2]) - char[3]</ExpectedInput>
        ///		<ExpectedOutput>HL7Exception</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="8918"> 
        ///		<ExpectedInput>Invalid ADT Message (missing and invalid segment data)</ExpectedInput>
        ///		<ExpectedOutput>Valid HL7ProtocolMessage with AR ack code</ExpectedOutput>
        ///	</Case>
        ///	
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// This method parses an incoming HL7 message for the Patient Update Interface,  
        /// saving patient demographic and\or location information that has changed in VistA.
        /// BR_111.01
        /// </summary>
        /// <param name="hl7Interface"><see cref="HL7Interface"/> containing interface parameters for the Patient Update HL7 interface.</param>
        /// <param name="message">Valid HL7 ADT message.</param>
        /// <returns>HL7ProtocolMessage Acknowledgement Message.</returns>
        public static HL7ProtocolMessage ParseHL7Message(HL7Interface hl7Interface, string message)
        {
            #region Handle Invalid Parameters

            if (hl7Interface == null)
            {
                throw (new ArgumentNullException("intParms"));
            }
            //
            if (message == null)
            {
                throw (new ArgumentNullException("message"));
            }

            #endregion
            //
            #region Variables

            string errData = null;
            string errMessage = null;
            //
            HL7ProtocolMessage ackMessage = null;
            //
            HL7AdtMessage hl7AdtMessage = null;
            //
            PatientUpdateData patientUpdateData = null;
            //
            Common.MessageStatus messageStatus = MessageStatus.ProcessingIncomingMessage;
            //
            DAL.HL7.OpenLibrary.AckCodes ackCode = AckCodes.AA;

            #endregion
            //
            try
            {
                hl7AdtMessage = new HL7AdtMessage(message);
                //
                if (hl7AdtMessage.AckErrorCode == AckErrorCodes.MessageAccepted)
                {
                    patientUpdateData = new PatientUpdateData();
                    //
                    errData = ValidateMessageContent(hl7Interface, hl7AdtMessage, ref patientUpdateData);
                    //
                    // Check message to ensure segment data has required fields
                    if (errData == string.Empty)
                    {
                        HL7MessageLog.InsertMessageLog(HL7MessageLog.GetMessageDataForMessageLogInsert(message, messageStatus, hl7Interface, Common.UpdateFunction.HL7PatientUpdateInterface));
                        //
                        ProcessUpdatedData(hl7AdtMessage, patientUpdateData);
                        //
                        messageStatus = MessageStatus.SuccessfullyCompleted;
                    }
                    else
                    {
                        ackCode = AckCodes.AR;
                        errMessage = string.Concat(ERROR_MESSAGE_BASE, ERROR_MESSAGE_HL7, errData);
                        messageStatus = MessageStatus.ProcessingError;
                    }
                }
                else
                {
                    ackCode = AckCodes.AR;
                    errData = AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], null, 0, 0, hl7AdtMessage.AckErrorCode, string.Empty);
                    errMessage = hl7AdtMessage.AckErrorCode.ToString();
                    messageStatus = MessageStatus.ProcessingError;
                }
            }
            catch (Exception ex)
            {
                ackCode = AckCodes.AR;
                errData = AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], null, 0, 0, AckErrorCodes.ApplicationInternalError, null);
                errMessage = string.Concat(ERROR_MESSAGE_BASE, HL7Utility.GetInnerExceptionFromException(ex));
                messageStatus = MessageStatus.ProcessingError;
            }
            //
            #region Handle Acknowledgement

            if (hl7AdtMessage != null)
            {
                ackMessage = AdtAckMessage.BuildApplicationAcknowledgement(hl7Interface, ackCode, hl7AdtMessage.GetMessageControlID(), errData, hl7AdtMessage.TriggerEvent); // CR 2985
                //
                HL7MessageLog.InsertAckMessageControlId(hl7AdtMessage.MessageControlID, ackMessage.GetMessageControlID(), messageStatus, hl7Interface.InterfaceName, Common.UpdateFunction.HL7PatientUpdateInterface);
                //
                if (ackCode == AckCodes.AR)
                {
                    HL7MessageLog.UpdateMessageStatus(messageStatus, hl7AdtMessage.GetMessageControlID(), errMessage, 0, hl7Interface.InterfaceName, Common.UpdateFunction.HL7PatientUpdateInterface);
                    //
                    //EventLogAppender
                    _eventsLogger.Error("VBECS Patient Update HL7 Parser: " + errMessage);
                    //System.Diagnostics.EventLog.WriteEntry("VBECS Patient Update HL7 Parser", errMessage, System.Diagnostics.EventLogEntryType.Error);
                    //
                    try
                    {
                        new HL7MailMessage().SendMessage(hl7Interface.InterfaceAdministratorEmail, PII                                  ", "Patient Update HL7 Parser Error", errMessage, "smtp.DNS   ");
                    }
                    catch (Exception mailexc)
                    {
                        string mailerrmsg = string.Concat("Unable to send mail message to Interface Administrator: ", HL7Utility.GetInnerExceptionFromException(mailexc));
                        //
                        //EventLogAppender
                        _eventsLogger.Error("VBECS Patient Update HL7 Parser: " + mailerrmsg);
                    }
                }
            }
            //
            return ackMessage;

            #endregion
        }

        #endregion

        #region Private Methods

        #region Validate Message

        /// <summary>
        /// 
        /// </summary>
        private static string ValidateMessageContent(HL7Interface hl7Interface, HL7AdtMessage hl7AdtMessage, ref PatientUpdateData patientUpdateData)
        {
            StringBuilder errSegments = new StringBuilder();
            //
            errSegments.Append(ValidateMshSegmentContent(hl7Interface, hl7AdtMessage));
            errSegments.Append(ValidateEvnSegmentContent(hl7Interface, hl7AdtMessage, ref patientUpdateData));
            errSegments.Append(ValidatePidSegmentContent(hl7Interface, hl7AdtMessage, ref patientUpdateData));
            errSegments.Append(ValidatePv1SegmentContent(hl7Interface, hl7AdtMessage, ref patientUpdateData));
            //
            return errSegments.ToString();
        }

        /// <summary>
        /// 
        /// </summary>
        private static string ValidateMshSegmentContent(HL7Interface hl7Interface, HL7AdtMessage hl7AdtMessage)
        {
            StringBuilder errSegment = new StringBuilder();
            //
            // CR 2985
            // Validate field separator (MSH[1]) and encoding characters (MSH[2]), these are critical for processing of message content.
            //
            // *********************************************************************************************
            // Field Separator: it can't be NULL or we would never get this far, but we can make sure it 
            // matches the defined value in the Interface table.
            if (hl7AdtMessage.MSH[1] != hl7Interface.FieldSeparator)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 1, AckErrorCodes.UnknownKeyIdentifier, string.Empty));
            }
            // *********************************************************************************************
            // Encoding Characters: these can't be NULL or we would never get this far, but we can make sure  
            // they match the defined value in the Interface table.
            char[] encodingCharacters = hl7AdtMessage.Message.ToCharArray(4, 4);
            //
            for (int idx = 0; idx < encodingCharacters.Length && idx < hl7Interface.EncodingCharacters.Length; idx++)
            {
                if (encodingCharacters[idx] != hl7Interface.EncodingCharacters[idx])
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 2, AckErrorCodes.UnknownKeyIdentifier, string.Empty));
                    break;
                }
            }
            // *********************************************************************************************
            // Sending Application
            if (hl7AdtMessage.MSH.Length < 4 || hl7AdtMessage.MSH[3] == null || hl7AdtMessage.MSH[3].Length == 0)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 3, AckErrorCodes.RequiredFieldMissing, string.Empty));
            }
            // *********************************************************************************************
            // Receiving Application
            if (hl7AdtMessage.MSH.Length < 6 || hl7AdtMessage.MSH[5] == null || hl7AdtMessage.MSH[5].Length == 0)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 5, AckErrorCodes.RequiredFieldMissing, string.Empty));
            }
            // *********************************************************************************************
            // Date/Time Of Message
            if (hl7AdtMessage.MSH.Length < 8 || hl7AdtMessage.MSH[7] == null || hl7AdtMessage.MSH[7].Length == 0)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 7, AckErrorCodes.RequiredFieldMissing, string.Empty));
            }
            // *********************************************************************************************
            // Message Type
            if (hl7AdtMessage.MSH.Length < 10 || hl7AdtMessage.MSH[9] == null || hl7AdtMessage.MSH[9].Length == 0)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 9, AckErrorCodes.RequiredFieldMissing, string.Empty));
            }
            else
            {
                if (hl7AdtMessage.MessageType == null)
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 9, AckErrorCodes.RequiredFieldMissing, string.Empty));
                }
                else if (!hl7AdtMessage.IsMessageTypeValid())
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 9, AckErrorCodes.UnsupportedMessageType, string.Empty));
                }
                //
                if (hl7AdtMessage.TriggerEvent == null)
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 9, AckErrorCodes.RequiredFieldMissing, string.Empty));
                }
                else if (!hl7AdtMessage.IsTriggerEventValid())
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 9, AckErrorCodes.UnsupportedMessageType, string.Empty));
                }
            }
            // *********************************************************************************************
            // Message Control ID
            if (hl7AdtMessage.MSH.Length < 11 || hl7AdtMessage.MSH[10] == null || hl7AdtMessage.MSH[10].Length == 0)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 10, AckErrorCodes.RequiredFieldMissing, string.Empty));
            }
            
            // *********************************************************************************************
            // Processing ID
           if (hl7AdtMessage.MSH.Length < 12 || hl7AdtMessage.MSH[11] == null || hl7AdtMessage.MSH[11].Length == 0)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 11, AckErrorCodes.RequiredFieldMissing, string.Empty));
            }
            //CR 3243
            else
            {
                char[] procId = hl7AdtMessage.MSH[11].ToCharArray();
                if (procId[0] != hl7Interface.ProcessingId)
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 11, AckErrorCodes.ConflictingProcessingId, string.Empty));
                }
            }             


            // *********************************************************************************************
            // Version ID
            if (hl7AdtMessage.MSH.Length < 13 || hl7AdtMessage.MSH[12] == null || hl7AdtMessage.MSH[12].Length == 0)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 12, AckErrorCodes.RequiredFieldMissing, string.Empty));
            }
            else if (hl7AdtMessage.MSH[12] != hl7Interface.VersionId)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "MSH", 1, 12, AckErrorCodes.UnsupportedVersionId, string.Empty));
            }
            //
            return errSegment.ToString();
        }

        /// <summary>
        /// 
        /// </summary>
        private static string ValidateEvnSegmentContent(HL7Interface hl7Interface, HL7AdtMessage hl7AdtMessage, ref PatientUpdateData patientUpdateData)
        {
            StringBuilder errSegment = new StringBuilder();
            //
            // Make sure EVN is present ... 
            if (hl7AdtMessage.EVN == null || hl7AdtMessage.EVN.Length == 0)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "EVN", 0, 0, AckErrorCodes.SegmentSequenceError, "EVN Segment data not found."));
            }
            else
            {
                // Operator ID: this field is declared optional in the message profile spec 
                if (hl7AdtMessage.EVN.Length > 5 && hl7AdtMessage.EVN[5] != null && hl7AdtMessage.EVN[5].Length > 0)
                {
                    patientUpdateData.OperatorId = hl7AdtMessage.EVN[5];
                }
            }
            //
            return errSegment.ToString();
        }

        /// <summary>
        ///  
        /// </summary>
        private static string ValidatePidSegmentContent(HL7Interface hl7Interface, HL7AdtMessage hl7AdtMessage, ref PatientUpdateData patientUpdateData)
        {
            StringBuilder errSegment = new StringBuilder();
            //
            // Make sure PID is present ... 
            if (hl7AdtMessage.PID == null || hl7AdtMessage.PID.Length == 0)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 0, 0, AckErrorCodes.SegmentSequenceError, "PID Segment data not found."));
            }
            else
            {
                // Patient ICN 
                if (hl7AdtMessage.PID.Length >= 3 && hl7AdtMessage.PID[2] != null && hl7AdtMessage.PID[2].Length > 0)
                {
                    patientUpdateData.PatientIcn = HL7Utility.ConvertString(hl7AdtMessage.PID[2]);
                }
                // Patient DFN (VistAPatientId) - this value cannot be changed in VistA
                if (hl7AdtMessage.PID.Length < 4 || hl7AdtMessage.PID[3] == null || hl7AdtMessage.PID[3].Length == 0)
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 3, AckErrorCodes.RequiredFieldMissing, string.Empty));	// CR 2979
                }
                else
                {
                    patientUpdateData.PatientDfn = hl7AdtMessage.PID[3].Split(hl7Interface.EncodingCharacters[0])[0];
                    //
                    if (patientUpdateData.PatientDfn == string.Empty || !Common.RegularExpressions.Integer().IsMatch(patientUpdateData.PatientDfn))
                    {
                        errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 3, AckErrorCodes.DataTypeError, string.Empty));
                    }
                }
                // *********************************************************************************************
                // Patient Name
                if (hl7AdtMessage.PID.Length < 6 || hl7AdtMessage.PID[5] == null || hl7AdtMessage.PID[5].Length == 0)
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 5, AckErrorCodes.RequiredFieldMissing, string.Empty));
                }
                else
                {
                    string[] patientName = hl7AdtMessage.PID[5].Split(hl7Interface.EncodingCharacters[0]);
                    //
                    patientUpdateData.PatientLastName = patientName.Length > 0 ? HL7Utility.ConvertString(patientName[0]) : null;
                    patientUpdateData.PatientFirstName = patientName.Length > 1 ? HL7Utility.ConvertString(patientName[1]) : null;
                    patientUpdateData.PatientMiddleName = patientName.Length > 2 ? HL7Utility.ConvertString(patientName[2]) : null;	// CR 2979
                    //
                    if (patientUpdateData.PatientLastName == null)
                    {
                        errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 5, AckErrorCodes.RequiredFieldMissing, string.Empty));
                    }
                    //
                    if (patientUpdateData.PatientFirstName == null)
                    {
                        errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 5, AckErrorCodes.RequiredFieldMissing, string.Empty));
                    }
                }
                // *********************************************************************************************
                // Patient DOB
                if (hl7AdtMessage.PID.Length < 8 || hl7AdtMessage.PID[7] == null || hl7AdtMessage.PID[7].Length == 0)
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 7, AckErrorCodes.RequiredFieldMissing, string.Empty));
                }
                else
                {
                    try
                    {
                        patientUpdateData.PatientDob = HL7DateFormat.ConvertHL7DateTime(hl7AdtMessage.PID[7]);
                        //CR 3237
                        if (patientUpdateData.PatientDob.ToShortDateString() == DateTime.MinValue.ToShortDateString())
                        {
                            errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 7, AckErrorCodes.DataTypeError, string.Empty));
                        }
                        //
                        if (hl7AdtMessage.PID[7].Substring(4, 4) == "0000")
                        {
                            patientUpdateData.PatientDobCode = "B";
                        }
                        else if (hl7AdtMessage.PID[7].Substring(4, 2) == "00")
                        {
                            patientUpdateData.PatientDobCode = "M";
                        }
                        else if (hl7AdtMessage.PID[7].Substring(6, 2) == "00")
                        {
                            patientUpdateData.PatientDobCode = "D";
                        }
                        else
                        {
                            patientUpdateData.PatientDobCode = "V";
                        }
                    }
                    catch (Exception)
                    {
                        errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 7, AckErrorCodes.DataTypeError, string.Empty));
                    }
                }
                // *********************************************************************************************
                // Patient Gender
                if (hl7AdtMessage.PID.Length < 9 || hl7AdtMessage.PID[8] == null || hl7AdtMessage.PID[8].Length == 0)
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 8, AckErrorCodes.RequiredFieldMissing, string.Empty));
                }
                else
                {
                    patientUpdateData.PatientSexCode = HL7Utility.ConvertString(hl7AdtMessage.PID[8]);
                }
                // *********************************************************************************************
                // Patient SSN
                if (hl7AdtMessage.PID.Length < 20 || hl7AdtMessage.PID[19] == null || hl7AdtMessage.PID[19].Length == 0)
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 19, AckErrorCodes.RequiredFieldMissing, string.Empty));
                }
                else
                {
                    patientUpdateData.PatientSsn = HL7Utility.ConvertString(hl7AdtMessage.PID[19]);
                    //
                    if (!Common.RegularExpressions.PatientSsn().IsMatch(patientUpdateData.PatientSsn))
                    {
                        errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 19, AckErrorCodes.UnknownKeyIdentifier, string.Empty));
                    }
                }
                // *********************************************************************************************
                // Patient DOD (optional)
                if (hl7AdtMessage.PID.Length > 29 && hl7AdtMessage.PID[29] != null && hl7AdtMessage.PID[29] != string.Empty)
                {
                    try
                    {
                        patientUpdateData.PatientDod = HL7DateFormat.ConvertHL7DateTime(hl7AdtMessage.PID[29]);
                        //CR 3237
                        if (patientUpdateData.PatientDod.ToShortDateString() == DateTime.MinValue.ToShortDateString())
                        {
                            errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 29, AckErrorCodes.DataTypeError, string.Empty));
                        }
                    }
                    catch (Exception)
                    {
                        errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PID", 1, 29, AckErrorCodes.DataTypeError, string.Empty));
                    }
                }
            }
            //
            return errSegment.ToString();
        }

        /// <summary>
        /// 
        /// </summary>
        private static string ValidatePv1SegmentContent(HL7Interface hl7Interface, HL7AdtMessage hl7AdtMessage, ref PatientUpdateData patientUpdateData)
        {
            StringBuilder errSegment = new StringBuilder();
            //
            // Make sure PV1 is present ... 
            if (hl7AdtMessage.PV1 == null || hl7AdtMessage.PV1.Length == 0)
            {
                errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PV1", 0, 0, AckErrorCodes.SegmentSequenceError, "PV1 Segment data not found."));
            }
            else
            {
                // Patient Class
                if (hl7AdtMessage.PV1.Length < 3 || hl7AdtMessage.PV1[2] == null || hl7AdtMessage.PV1[2].Length == 0)
                {
                    errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PV1", 1, 2, AckErrorCodes.RequiredFieldMissing, string.Empty));
                }
                else
                {
                    if (hl7AdtMessage.PV1[2].Length > 1 && (hl7AdtMessage.PV1[2] != "I" || hl7AdtMessage.PV1[2] != "O"))
                    {
                        errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PV1", 1, 2, AckErrorCodes.UnknownKeyIdentifier, string.Empty));
                    }
                    else
                    {
                        // Discharges need to be recorded as outpatient
                        patientUpdateData.InpatientIndicator = hl7AdtMessage.PV1[2] == "I" && !hl7AdtMessage.IsDischarge() ? true : false;
                    }
                }
                // *********************************************************************************************
                // Assigned Patient Location
                if (hl7AdtMessage.PV1.Length < 4 || hl7AdtMessage.PV1[3] == null || hl7AdtMessage.PV1[3].Length == 0)
                {
                    //per the VAIS the assigned patient location is not a required field.
                    //CR 3232
                    return errSegment.ToString();
                }
                else
                {
                    string[] locationData = hl7AdtMessage.PV1[3].Split(hl7Interface.EncodingCharacters[0]);
                    //
                    patientUpdateData.LocationName = GetLocationDataFromArray(locationData, 0);		// PV1.3.1
                    // 
                    // CR 2970: Room and Bed are now stored in one field (optional)
                    string patientRoom = GetLocationDataFromArray(locationData, 1);					// PV1.3.2
                    string patientBed = GetLocationDataFromArray(locationData, 2);					// PV1.3.3
                    if (patientRoom != null && patientRoom.Length > 0
                        && patientBed != null && patientBed.Length > 0)
                    {
                        patientRoom += "-";
                    }
                    patientUpdateData.LocationRoomBed = patientRoom + patientBed;
                    //
                    patientUpdateData.LocationDivisionCode = hl7AdtMessage.PV1.Length > 39 ? hl7AdtMessage.PV1[39] : null;		// PV1.39
                    //
                    if (patientUpdateData.InpatientIndicator && (patientUpdateData.LocationName == null || patientUpdateData.LocationName.Length == 0))
                    {
                        errSegment.Append(AdtAckMessage.BuildErrSegment(hl7Interface.FieldSeparator, hl7Interface.EncodingCharacters[0], "PV1", 1, 3, AckErrorCodes.RequiredFieldMissing, string.Empty));
                    }
                }
            }
            //
            return errSegment.ToString();
        }

        #endregion

        #region Process Updated Patient Data

        /// <summary>
        /// 
        /// Process patient updates
        /// </summary>
        private static void ProcessUpdatedData(HL7AdtMessage hl7AdtMessage, PatientUpdateData patientUpdateData)
        {
            ArrayList storedProcedureList = new ArrayList();
            ArrayList updatedTableList = new ArrayList();
            //
            // CR 2979
            if (hl7AdtMessage.IsDemographicUpdate())
            {
                GetUpdatedPatientDemographicData(ref storedProcedureList, ref updatedTableList, patientUpdateData);
            }
            else
            {
                // If there are patient demographic updates _patientGuid will be set
                // in GetUpdatedPatientDemographicData(), otherwise we must look it up here
                patientUpdateData.PatientGuid = DAL.Patient.GetPatientGuidForPatient(Convert.ToInt64(patientUpdateData.PatientDfn), null);
            }
            // 
            GetUpdatedPatientLocationData(ref storedProcedureList, ref updatedTableList, patientUpdateData, hl7AdtMessage.IsDischarge());
            //
            if (storedProcedureList != null && storedProcedureList.Count > 0 && updatedTableList != null && updatedTableList.Count > 0)
            {
                HL7AL.PatientChange.UpdatePatientInformation(storedProcedureList, updatedTableList);
            }
        }

        /// <summary> 
        /// Compare new patient demographic data with existing;
        /// save any changes
        /// </summary>
        private static void GetUpdatedPatientDemographicData(ref ArrayList storedProcedureList, ref ArrayList updatedTableList, PatientUpdateData patientUpdateData)
        {
            DataRow drFromPatient = HL7AL.CprsOmgMessage.GetHl7Patient(null, patientUpdateData.PatientDfn).Rows[0];
            //
            if (drFromPatient != null)
            {
                bool demographicUpdatesAvailable = false;
                //
                patientUpdateData.PatientGuid = (Guid)drFromPatient[PATIENT.PatientGuid];
                //
                #region Initialize DataTables\DataRows

                DataTable dtPatientChange = HL7AL.PatientChange.GetPatientChangeDataTable();
                DataRow drPatientChange = dtPatientChange.NewRow();
                //
                drPatientChange[PATIENT_CHANGE.PatientChangeStatusCode] = "P";
                drPatientChange[PATIENT_CHANGE.PatientChangeGuid] = System.Guid.NewGuid();
                drPatientChange[PATIENT_CHANGE.FromPatientGuid] = patientUpdateData.PatientGuid;
                drPatientChange[PATIENT_CHANGE.ToPatientGuid] = patientUpdateData.PatientGuid;
                //
                DataTable dtPatientUpdate = HL7AL.PatientChange.GetPatientUpdateDataTable();
                DataRow drPatientUpdate = dtPatientUpdate.NewRow();
                //
                foreach (System.Data.DataColumn dc in dtPatientUpdate.Columns)
                {
                    if (drFromPatient.Table.Columns.Contains(dc.ColumnName))
                    {
                        drPatientUpdate[dc.ColumnName] = drFromPatient[dc.ColumnName];
                    }
                }

                #endregion
                //
                #region Patient Icn

                // ****************************************************************************************************************************************
                // **** HANDLE PATIENT ICN UPDATES ***
                // ************************************
                if (patientUpdateData.PatientIcn != drFromPatient[PATIENT.PatientIcn].ToString())
                {
                    demographicUpdatesAvailable = true;
                    //
                    drPatientChange[PATIENT_CHANGE.ToPatientIcn] = patientUpdateData.PatientIcn;
                    drPatientUpdate[PATIENT.PatientIcn] = patientUpdateData.PatientIcn;
                    drPatientChange[PATIENT_CHANGE.FromPatientIcn] = drFromPatient[PATIENT.PatientIcn].ToString();
                }

                #endregion
                //
                #region Patient Name

                // ****************************************************************************************************************************************
                // **** HANDLE PATIENT NAME UPDATES ***
                // ************************************* 
                if (patientUpdateData.PatientLastName != drFromPatient[PATIENT.PatientLastName].ToString())
                {
                    demographicUpdatesAvailable = true;
                    //
                    drPatientChange[PATIENT_CHANGE.ToPatientLastName] = patientUpdateData.PatientLastName;
                    drPatientUpdate[PATIENT.PatientLastName] = patientUpdateData.PatientLastName;
                    drPatientChange[PATIENT_CHANGE.FromPatientLastName] = drFromPatient[PATIENT.PatientLastName];
                }
                //
                if (patientUpdateData.PatientFirstName != drFromPatient[PATIENT.PatientFirstName].ToString())
                {
                    demographicUpdatesAvailable = true;
                    //
                    drPatientChange[PATIENT_CHANGE.ToPatientFirstName] = patientUpdateData.PatientFirstName;
                    drPatientUpdate[PATIENT.PatientFirstName] = patientUpdateData.PatientFirstName;
                    drPatientChange[PATIENT_CHANGE.FromPatientFirstName] = drFromPatient[PATIENT.PatientFirstName];
                }
                // CR 3315
                //if the message is nulling out patient middle name check if the db also has null
                //message has null
                if (patientUpdateData.PatientMiddleName == null)
                {
                    //DB doesnt have null
                    if (!drFromPatient.IsNull(PATIENT.PatientMiddleName))
                    {
                        demographicUpdatesAvailable = true;
                        //
                        drPatientChange[PATIENT_CHANGE.ToPatientMIddleName] = patientUpdateData.PatientMiddleName;
                        drPatientUpdate[PATIENT.PatientMiddleName] = patientUpdateData.PatientMiddleName;
                        drPatientChange[PATIENT_CHANGE.FromPatientMIddleName] = drFromPatient[PATIENT.PatientMiddleName];
                    }
                }
                //message doesnt contain a null middle name
                else
                {
                    //Check if the DB has a middle name
                    if (drFromPatient.IsNull(PATIENT.PatientMiddleName))
                    {
                        demographicUpdatesAvailable = true;
                        //
                        drPatientChange[PATIENT_CHANGE.ToPatientMIddleName] = patientUpdateData.PatientMiddleName;
                        drPatientUpdate[PATIENT.PatientMiddleName] = patientUpdateData.PatientMiddleName;
                        drPatientChange[PATIENT_CHANGE.FromPatientMIddleName] = drFromPatient[PATIENT.PatientMiddleName];
                    }
                    else
                    {//neither are null in this case so check if it has changed
                        if (patientUpdateData.PatientMiddleName != drFromPatient[PATIENT.PatientMiddleName].ToString())
                        {
                            demographicUpdatesAvailable = true;
                            //
                            drPatientChange[PATIENT_CHANGE.ToPatientMIddleName] = patientUpdateData.PatientMiddleName;
                            drPatientUpdate[PATIENT.PatientMiddleName] = patientUpdateData.PatientMiddleName;
                            drPatientChange[PATIENT_CHANGE.FromPatientMIddleName] = drFromPatient[PATIENT.PatientMiddleName];
                        }
                    }
                }

                #endregion
                //	
                #region Patient Date of Birth

                // ****************************************************************************************************************************************
                // **** HANDLE PATIENT DOB UPDATES ***
                // ************************************
                //CR 3237
                //CR 3315
                System.DateTime dtCompareDOB = System.DateTime.MinValue;
                if (drFromPatient[PATIENT.PatientDob] != System.DBNull.Value)
                {
                    dtCompareDOB = (System.DateTime)(drFromPatient[PATIENT.PatientDob]);
                }
                if (patientUpdateData.PatientDob.ToShortDateString() != dtCompareDOB.ToShortDateString())
                {
                    demographicUpdatesAvailable = true;
                    //
                    drPatientChange[PATIENT_CHANGE.ToPatientDob] = patientUpdateData.PatientDob.ToShortDateString();
                    drPatientUpdate[PATIENT.PatientDob] = patientUpdateData.PatientDob.ToShortDateString();
                    drPatientChange[PATIENT_CHANGE.FromPatientDob] = drFromPatient[PATIENT.PatientDob];
                    //
                    // *************************************************************************************************************************************
                    // **** HANDLE PATIENT DOB CODE UPDATES ***
                    // *****************************************
                    drPatientUpdate[PATIENT.PatientDobCode] = patientUpdateData.PatientDobCode;
                }

                #endregion
                //
                #region Patient Ssn

                // ****************************************************************************************************************************************
                // **** HANDLE PATIENT SSN UPDATES ***
                // ************************************
                if (patientUpdateData.PatientSsn != drFromPatient[PATIENT.PatientSsn].ToString())
                {
                    demographicUpdatesAvailable = true;
                    //
                    if (patientUpdateData.PatientSsn.EndsWith("P"))
                    {
                        drPatientChange[PATIENT_CHANGE.ToSsnPseudoIndicator] = 1;
                        drPatientUpdate[PATIENT.SsnPseudoIndicator] = 1;
                        drPatientChange[PATIENT_CHANGE.ToPatientSsn] = patientUpdateData.PatientSsn.Remove(patientUpdateData.PatientSsn.Length - 1, 1);
                        drPatientUpdate[PATIENT.PatientSsn] = patientUpdateData.PatientSsn.Remove(patientUpdateData.PatientSsn.Length - 1, 1);
                    }
                    else
                    {
                        drPatientChange[PATIENT_CHANGE.ToPatientSsn] = patientUpdateData.PatientSsn;
                        drPatientUpdate[PATIENT.PatientSsn] = patientUpdateData.PatientSsn;
                        drPatientChange[PATIENT_CHANGE.ToSsnPseudoIndicator] = 0;
                        drPatientUpdate[PATIENT.SsnPseudoIndicator] = 0;
                    }
                    //
                    drPatientChange[PATIENT_CHANGE.FromPatientSsn] = drFromPatient[PATIENT.PatientSsn].ToString();
                    drPatientChange[PATIENT_CHANGE.FromSsnPseudoIndicator] = drFromPatient[PATIENT.SsnPseudoIndicator];
                }

                #endregion
                //		
                #region Patient Sex Code

                // ****************************************************************************************************************************************
                // **** HANDLE PATIENT SEX CODE UPDATES ***
                // *****************************************
                if (patientUpdateData.PatientSexCode != drFromPatient[PATIENT.PatientSexCode].ToString())
                {
                    demographicUpdatesAvailable = true;
                    //
                    drPatientChange[PATIENT_CHANGE.ToPatientSexCode] = patientUpdateData.PatientSexCode;
                    drPatientUpdate[PATIENT.PatientSexCode] = patientUpdateData.PatientSexCode;
                    drPatientChange[PATIENT_CHANGE.FromPatientSexCode] = drFromPatient[PATIENT.PatientSexCode].ToString();
                }

                #endregion
                //
                #region Optional: Patient Date of Death

                // ****************************************************************************************************************************************
                // **** HANDLE PATIENT DOD UPDATES ***
                // ************************************
                // CR 2974: Handle both new date notifications and updates to existing values
                //if the HL7 message is not setting DOB to null
                if (patientUpdateData.PatientDod.ToShortDateString() != DateTime.MinValue.ToShortDateString())
                {
                    // If date is not the same or from date was null ...
                    //CR 3237
                    //If the current database patient death dates is not null, check if the message has a different DOD, otherwise if the 
                    //Current DOD in the database is null 
                    System.DateTime dtCompareDOD;
                    if (!drFromPatient.IsNull(PATIENT.PatientDeathDate))
                    {
                        dtCompareDOD = (System.DateTime)(drFromPatient[PATIENT.PatientDeathDate]);
                    }
                    else
                    {
                        dtCompareDOD = DateTime.MinValue;
                    }
                    if (!drFromPatient.IsNull(PATIENT.PatientDeathDate) && dtCompareDOD.ToShortDateString() != patientUpdateData.PatientDod.ToShortDateString()
                        || drFromPatient.IsNull(PATIENT.PatientDeathDate))
                    {
                        demographicUpdatesAvailable = true;
                        //
                        drPatientChange[PATIENT_CHANGE.ToPatientDeathDate] = patientUpdateData.PatientDod.ToShortDateString();
                        drPatientUpdate[PATIENT.PatientDeathDate] = patientUpdateData.PatientDod.ToShortDateString();
                        //
                        drPatientChange[PATIENT_CHANGE.FromPatientDeathDate] = drFromPatient[PATIENT.PatientDeathDate];
                    }
                }
                else if (patientUpdateData.PatientDod.ToShortDateString() == DateTime.MinValue.ToShortDateString() && !drFromPatient.IsNull(PATIENT.PatientDeathDate))
                {
                    // CR 1826: If death date is null in message, set it to null in the db.
                    //
                    demographicUpdatesAvailable = true;
                    //
                    drPatientChange[PATIENT_CHANGE.ToPatientDeathDate] = System.DBNull.Value;
                    drPatientUpdate[PATIENT.PatientDeathDate] = System.DBNull.Value;
                    //
                    drPatientChange[PATIENT_CHANGE.FromPatientDeathDate] = drFromPatient[PATIENT.PatientDeathDate];
                }

                #endregion
                //
                #region If there are updates, add DataTables to _updatedTableList

                if (demographicUpdatesAvailable)
                {
                    string lastUpdateUser = LAST_UPDATE_USER_VALUE;
                    //
                    if (patientUpdateData.OperatorId != null && patientUpdateData.OperatorId != string.Empty)
                    {
                        lastUpdateUser = patientUpdateData.OperatorId;
                    }
                    //
                    drPatientUpdate[PATIENT.LastUpdateUser] = lastUpdateUser;
                    drPatientUpdate[PATIENT.LastUpdateFunctionId] = Common.UpdateFunction.HL7PatientUpdateInterface;
                    dtPatientUpdate.Rows.Add(drPatientUpdate);
                    //
                    drPatientChange[PATIENT_CHANGE.LastUpdateUser] = lastUpdateUser;
                    drPatientChange[PATIENT_CHANGE.LastUpdateFunctionId] = Common.UpdateFunction.HL7PatientUpdateInterface;
                    dtPatientChange.Rows.Add(drPatientChange);
                    //
                    storedProcedureList.Add(SProc.HL7InsertPatientChange.StoredProcName);
                    storedProcedureList.Add(SProc.HL7UpdatePatient.StoredProcName);
                    //
                    updatedTableList.Add(dtPatientChange.Copy());
                    updatedTableList.Add(dtPatientUpdate.Copy());
                }

                #endregion
            }
        }

        /// <summary>
        /// 
        /// </summary>
        private static void GetUpdatedPatientLocationData(ref ArrayList storedProcedureList, ref ArrayList updatedTableList, PatientUpdateData patientUpdateData, bool isDischarge)
        {
            if (patientUpdateData.PatientGuid != Guid.Empty)
            {
                HL7AL.PatientLocation patientLocation = HL7AL.PatientLocation.GetHL7PatientLocation(patientUpdateData.PatientGuid);
                //
                bool isUpdate = false;
                bool locationChanged = false;
                //
                if (patientLocation != null)
                {
                    //
                    // See if anything changed ...
                    if (patientLocation.InPatientIndicator != patientUpdateData.InpatientIndicator
                        || !Equals(patientLocation.DivisionCode, patientUpdateData.LocationDivisionCode)
                        || !Equals(patientLocation.Name, patientUpdateData.LocationName)
                        || !Equals(patientLocation.RoomBed, patientUpdateData.LocationRoomBed))
                    {
                        //CR 3315
                        isUpdate = true;
                        locationChanged = true;
                    }
                }
                //
                if (!isUpdate || isUpdate && locationChanged)
                {
                    string lastUpdateUser = LAST_UPDATE_USER_VALUE;
                    //
                    if (patientUpdateData.OperatorId != null && patientUpdateData.OperatorId != string.Empty)
                    {
                        lastUpdateUser = patientUpdateData.OperatorId;
                    }
                    //
                    DataTable dtPatientLocation = HL7AL.PatientLocation.GetPatientLocationDataTable();
                    DataRow drPatientLocation = dtPatientLocation.NewRow();
                    //
                    drPatientLocation[PATIENT_LOCATION.PatientGuid] = patientUpdateData.PatientGuid;
                    drPatientLocation[PATIENT_LOCATION.InPatientIndicator] = patientUpdateData.InpatientIndicator;
                    //
                    // Discharges: the location message will contain the Hospital Location from which patient was discharged; 
                    // however, the patient has left the building so we need to store nulls in VBECS
                    if (patientUpdateData.InpatientIndicator && !isDischarge)
                    {
                        drPatientLocation[PATIENT_LOCATION.LocationName] = patientUpdateData.LocationName;
                        // 
                        // CR 2970: Room and Bed are now stored in one field
                        drPatientLocation[PATIENT_LOCATION.LocationRoomBed] = patientUpdateData.LocationRoomBed;
                        drPatientLocation[PATIENT_LOCATION.DivisionCode] = patientUpdateData.LocationDivisionCode;
                    }
                    //
                    drPatientLocation[PATIENT_LOCATION.LastUpdateFunctionId] = Common.UpdateFunction.HL7PatientUpdateInterface;
                    drPatientLocation[PATIENT_LOCATION.LastUpdateUser] = lastUpdateUser;
                    //
                    dtPatientLocation.Rows.Add(drPatientLocation);
                    //
                    if (isUpdate)
                    {
                        storedProcedureList.Add(SProc.HL7UpdatePatientLocation.StoredProcName);
                    }
                    else
                    {
                        storedProcedureList.Add(SProc.HL7InsertPatientLocation.StoredProcName);
                    }
                    //
                    updatedTableList.Add(dtPatientLocation.Copy());
                }
            }
        }

        #endregion

        #region Helper Methods

        /// <summary>
        ///  
        /// </summary>
        private static bool Equals(string str1, string str2)
        {
            str1 = str1 != null ? str1.Trim() : null;
            //
            str2 = str2 != null ? str2.Trim() : null;
            //
            return str1 == str2;
        }


        /// <summary>
        ///  
        /// </summary>
        private static string GetLocationDataFromArray(string[] locationData, int index)
        {
            if (locationData != null
                && locationData.Length >= (index + 1)
                && locationData[index] != null
                && locationData[index].Length > 0
                && locationData[index] != EMPTY_LOCATION_1
                && locationData[index] != EMPTY_LOCATION_2)
            {
                return locationData[index];
            }
            else
            {
                return null;
            }
        }

        #endregion

        #endregion
    }
}